function [S,I,R]=sir_sim(beta,gamma,I0,V,target)
%
% function [S,I,R]=sir_sim(beta,gamma,I0,V,target)
%
% runs a simulation of an SIR model
%
% S: susceptible
% I: infective
% R: removed
%
% beta is the infectivity parameter, scaled so that N=1
% gamma is the reciprocal of mean recovery time
% I0 is the initial infectious fraction
% V is the initial immune fraction
% target is the infected fraction used as the end condition
%
% results is a matrix: columns are SIR, rows are 0:days
% 
% by Glenn Ledder
% written 2020/06/21
% revised 2020/10/26
%
% direct comments to gledder@unl.edu

%% DATA

% suggested default values
% beta = 0.5;
% gamma = 0.2;
% I0 = 0.01;
% V = 0;
% target = 0.001;

maxdays = 1000;

%% INITIALIZATION

% set up results data structure with Y=[S,I] and R

results = zeros(maxdays+1,3);
S0 = 1-I0-V;
Y = [S0,I0];
R = V;
results(1,:) = [Y,R];

y = Y';
oldi = I0;

%% COMPUTATION

for t=1:maxdays
    % y is a column vector, Y^T
    y = rk4(1,y);
    Y = y';
    R = 1-sum(Y);
    results(t+1,:) = [Y,R];
    if Y(2)>min(target,oldi)
        oldi = Y(2);
    else
        results = results(1:(t+1),:);
        break;
    end
end

S = results(:,1);
I = results(:,2);
R = results(:,3);

%% FUNCTION FOR rk4

    function y=rk4(dt,y0)
        % y0 is a column vector of initial conditions at t
        % y is a column vector of values at t+dt
        k1 = yprime(y0);
        k2 = yprime(y0+0.5*dt*k1);
        k3 = yprime(y0+0.5*dt*k2);
        k4 = yprime(y0+dt*k3);
        y = y0+dt*(k1+2*k2+2*k3+k4)/6;
    end
        
%% FUNCTION FOR THE DIFFERENTIAL EQUATION

    function yp=yprime(y)
    % split out components
        S = y(1);
        I = y(2);
    % compute derivatives
        Sp = -beta*S*I;
        Ip = -Sp-gamma*I;
    % assemble derivative
        yp = [Sp;Ip];
    end

%% END

end
